<?php
// --------------------------------------------------------------------------------
// PhpConcept Library - Zip Module 1.1.2
// --------------------------------------------------------------------------------
// License GNU/LGPL - Vincent Blavet - September 2002
// http://www.phpconcept.net
// --------------------------------------------------------------------------------
//
// Presentation :
//   PclZip is a PHP library that manage ZIP archives.
//   So far tests show that archives generated by PclZip are readable by
//   WinZip application and other tools.
//
// Description :
//   See readme.txt and http://www.phpconcept.net
//
// Warning :
//   This library and the associated files are non commercial, non professional
//   work.
//   It should not have unexpected results. However if any damage is caused by
//   this software the author can not be responsible.
//   The use of this software is at the risk of the user.
//
// --------------------------------------------------------------------------------

// ----- Look for double include
if (!defined("PCL_ZIP"))
{
  define( "PCL_ZIP", 1 );

  // ----- Error codes
  //   -1 : Unable to open file in binary write mode
  //   -2 : Unable to open file in binary read mode
  //   -3 : Invalid parameters
  //   -4 : File does not exist
  //   -5 : Filename is too long (max. 255)
  //   -6 : Not a valid zip file
  //   -7 : Invalid extracted file size
  //   -8 : Unable to create directory
  //   -9 : Invalid archive extension
  //  -10 : Invalid archive format
  //  -11 : Unable to delete file (unlink)
  //  -12 : Unable to rename file (rename)
  //  -13 : Invalid header checksum
  //  -14 : Invalid archive size

// --------------------------------------------------------------------------------
// ***** UNDER THIS LINE NOTHING NEEDS TO BE MODIFIED *****
// --------------------------------------------------------------------------------

  // ----- Global variables
  $g_pclzip_version = "1.1.2";

  // ----- Include other libraries
  // This library should be called by each script before the include of PhpZip
  // Library in order to limit the potential 'include' directory path problem.
  // By default these libraries are searched in the local path, and the include
  // path.
  if (!defined("PCLTRACE_LIB"))
  {
    include('pcltrace.lib.php');
  }
  if (!defined("PCLERROR_LIB"))
  {
    include('pclerror.lib.php');
  }

  // ----- Constants
  define( 'PCLZIP_READ_BLOCK_SIZE', 2048 );

  // ----- Optional static temporary directory
  //       By default temporary files are generated in the script current
  //       path.
  //       If defined :
  //       - MUST BE terminated by a '/'.
  //       - MUST be a valid, already created directory
  //       Samples :
  // define( 'PCLZIP_TEMPORARY_DIR', '/temp/' );
  // define( 'PCLZIP_TEMPORARY_DIR', 'C:/Temp/' );
  define( 'PCLZIP_TEMPORARY_DIR', 'compress' );

  // --------------------------------------------------------------------------------
  // Class : PclZip
  // Description :
  //   PclZip is the class that represent a Zip archive.
  //   The public methods allow the manipulation of the archive.
  // Attributes :
  //   Attributes must not be accessed directly.
  // Methods :
  //   PclZip() : Object creator
  //   create() : Creates the Zip archive
  //   listContent() : List the content of the Zip archive
  //   extract() : Extract the content of the archive
  //   properties() : List the properties of the archive
  // --------------------------------------------------------------------------------
  class PclZip
  {
    // ----- Filename of the zip file
    var $zipname = '';

    // ----- File descriptor of the zip file
    var $zip_fd = 0;

  // --------------------------------------------------------------------------------
  // Function : PclZip()
  // Description :
  //   Creates a PclZip object and set the name of the associated Zip archive
  //   filename.
  //   Note that no real action is taken, if the archive does not exist it is not
  //   created. Use create() for that.
  // --------------------------------------------------------------------------------
  function PclZip($p_zipname)
  {
    PclTraceFctStart(__FILE__, __LINE__, 'PclZip::PclZip', "zipname=$p_zipname");

    // ----- Tests the zlib
    if (!function_exists('gzopen'))
    {
      PclTraceFctMessage(__FILE__, __LINE__, 1, "zlib extension seems to be missing");
      die('Abort '.basename(__FILE__).' : Missing zlib extensions');
    }

    // ----- Set the attributes
    $this->zipname = $p_zipname;
    $this->zip_fd = 0;

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, 1);
    return;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : create()
  // Description :
  //   This method creates a Zip Archive. The Zip file is created in the
  //   filesystem. The files and directories indicated in $p_filelist
  //   are added in the archive. See the parameters description for the
  //   supported format of $p_filelist.
  //   When a directory is in the list, the directory and its content is added
  //   in the archive.
  // Parameters :
  //   $p_filelist : An array containing file or directory names, or
  //                 a string containing one filename or one directory name, or
  //                 a string containing a list of filenames and/or directory
  //                 names separated by spaces.
  //   $p_add_dir : A path to add before the real path of the archived file,
  //                in order to have it memorized in the archive.
  //   $p_remove_dir : A path to remove from the real path of the file to archive,
  //                   in order to have a shorter path memorized in the archive.
  //                   When $p_add_dir and $p_remove_dir are set, $p_remove_dir
  //                   is removed first, before $p_add_dir is added.
  // Return Values :
  //   0 on failure,
  //   The list of the added files, with a status of the add action.
  //   (see PclZip::listContent() for list entry format)
  // --------------------------------------------------------------------------------
  function create($p_filelist, $p_add_dir="", $p_remove_dir="")
  {
    PclTraceFctStart(__FILE__, __LINE__, 'PclZip::create', "filelist='$p_filelist',add_dir='$p_add_dir', remove_dir='$p_remove_dir'");
    $v_result=1;

    // ----- Look if the $p_filelist is really an array
    $p_result_list = array();
    if (is_array($p_filelist))
    {
      // ----- Call the create fct
      $v_result = $this->privCreate($p_filelist, $p_result_list, $p_add_dir, $p_remove_dir);
    }

    // ----- Look if the $p_filelist is a string
    else if (is_string($p_filelist))
    {
      // ----- Create a list with the elements from the string
      $v_list = explode(" ", $p_filelist);

      // ----- Call the create fct
      $v_result = $this->privCreate($v_list, $p_result_list, $p_add_dir, $p_remove_dir);
    }

    // ----- Invalid variable
    else
    {
      // ----- Error log
      PclErrorLog(-3, "Invalid variable type p_filelist");
      $v_result = -3;
    }

    if ($v_result != 1)
    {
      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, 0);
      return 0;
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $p_result_list);
    return $p_result_list;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : add()
  // Description :
  //   This methods add the list of files in an existing archive.
  //   If a file with the same name already exists, it is added at the end of the
  //   archive, the first one is still present.
  //   If the archive does not exist, it is created.
  // Parameters :
  //   $p_filelist : An array containing file or directory names, or
  //                 a string containing one filename or one directory name, or
  //                 a string containing a list of filenames and/or directory
  //                 names separated by spaces.
  //   $p_add_dir : A path to add before the real path of the archived file,
  //                in order to have it memorized in the archive.
  //   $p_remove_dir : A path to remove from the real path of the file to archive,
  //                   in order to have a shorter path memorized in the archive.
  //                   When $p_add_dir and $p_remove_dir are set, $p_remove_dir
  //                   is removed first, before $p_add_dir is added.
  // Return Values :
  //   0 on failure,
  //   The list of the added files, with a status of the add action.
  //   (see PclZip::listContent() for list entry format)
  // --------------------------------------------------------------------------------
  function add($p_filelist, $p_add_dir="", $p_remove_dir="")
  {
    PclTraceFctStart(__FILE__, __LINE__, 'PclZip::add', "filelist='$p_filelist',add_dir='$p_add_dir', remove_dir='$p_remove_dir'");
    $v_result=1;

    // ----- Look if the $p_filelist is really an array
    $p_result_list = array();
    if (is_array($p_filelist))
    {
      // ----- Call the create fct
      $v_result = $this->privAdd($p_filelist, $p_result_list, $p_add_dir, $p_remove_dir);
    }

    // ----- Look if the $p_filelist is a string
    else if (is_string($p_filelist))
    {
      // ----- Create a list with the elements from the string
      $v_list = explode(" ", $p_filelist);

      // ----- Call the create fct
      $v_result = $this->privAdd($v_list, $p_result_list, $p_add_dir, $p_remove_dir);
    }

    // ----- Invalid variable
    else
    {
      // ----- Error log
      PclErrorLog(-3, "Invalid variable type p_filelist");
      $v_result = -3;
    }

    if ($v_result != 1)
    {
      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, 0);
      return 0;
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $p_result_list);
    return $p_result_list;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : listContent()
  // Description :
  //   This public method, gives the list of the files and directories, with their
  //   properties.
  //   The properties of each entries in the list are (used also in other functions) :
  //     filename : Name of the file. For a create or add action it is the filename
  //                given by the user. For an extract function it is the filename
  //                of the extracted file.
  //     stored_filename : Name of the file / directory stored in the archive.
  //     size : Size of the stored file.
  //     compressed_size : Size of the file's data compressed in the archive
  //                       (without the headers overhead)
  //     mtime : Last known modification date of the file (UNIX timestamp)
  //     comment : Comment associated with the file
  //     folder : true | false
  //     index : index of the file in the archive
  //     status : status of the action (depending of the action) :
  //              Values are :
  //                ok : OK !
  //                filtered : the file / dir is not extracted (filtered by user)
  //                already_a_directory : the file can not be extracted because a
  //                                      directory with the same name already exists
  //                write_protected : the file can not be extracted because a file
  //                                  with the same name already exists and is
  //                                  write protected
  //                newer_exist : the file was not extracted because a newer file exists
  //                path_creation_fail : the file is not extracted because the folder
  //                                     does not exists and can not be created
  //                write_error : the file was not extracted because there was a
  //                              error while writing the file
  //                read_error : the file was not extracted because there was a error
  //                             while reading the file
  //                invalid_header : the file was not extracted because of an archive
  //                                 format error (bad file header)
  //   Note that each time a method can continue operating when there
  //   is an action error on a file, the error is only logged in the file status.
  // Return Values :
  //   0 on an unrecoverable failure,
  //   The list of the files in the archive.
  // --------------------------------------------------------------------------------
  function listContent()
  {
    PclTraceFctStart(__FILE__, __LINE__, 'PclZip::listContent', "");
    $v_result=1;

    // ----- Call the extracting fct
    $p_list = array();
    if (($v_result = $this->privList($p_list)) != 1)
    {
      unset($p_list);
      PclTraceFctEnd(__FILE__, __LINE__, 0, PclErrorString());
      return(0);
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $p_list);
    return $p_list;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : extract()
  // Description :
  //   This method extract all the files / directories from the archive to the
  //   folder indicated in $p_path.
  //   If you want to ignore the 'root' part of path of the memorized files
  //   you can indicate this in the optional $p_remove_path parameter.
  //   By default, if a newer file with the same name already exists, the
  //   file is not extracted.
  // Parameters :
  //   $p_path : Path where the files and directories are to be extracted
  //   $p_remove_path : First part ('root' part) of the memorized path
  //                    (if any similar) to remove while extracting.
  // Return Values :
  //   0 on failure,
  //   The list of the extracted files, with a status of the action.
  //   (see PclZip::listContent() for list entry format)
  // --------------------------------------------------------------------------------
  function extract($p_path="./", $p_remove_path="")
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip:extract", "path='$p_path', remove_path='$p_remove_path'");
    $v_result=1;

    // ----- Call the extracting fct
    $p_list = array();
    if (($v_result = $this->privExtract(&$p_list, $p_path, $p_remove_path)) != 1)
    {
      unset($p_list);
      PclTraceFctEnd(__FILE__, __LINE__, 0, PclErrorString());
      return(0);
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $p_list);
    return $p_list;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : extractByIndex()
  // Description :
  //   This method is doing a partial extract of the archive.
  //   The extracted files or folders are identified by their index in the
  //   archive (from 0 to n).
  //   Note that if the index identify a folder, only the folder entry is
  //   extracted, not all the files included in the archive.
  // Parameters :
  //   $p_index : A single index (integer) or a string of indexes of files to
  //              extract. The form of the string is "0,4-6,8-12" with only numbers
  //              and '-' for range or ',' to separate ranges. No spaces or ';'
  //              are allowed.
  //   $p_path : Path where the files and directories are to be extracted
  //   $p_remove_path : First part ('root' part) of the memorized path
  //                    (if any similar) to remove while extracting.
  // Return Values :
  //   0 on failure,
  //   The list of the extracted files, with a status of the action.
  //   (see PclZip::listContent() for list entry format)
  // --------------------------------------------------------------------------------
  function extractByIndex($p_index, $p_path="./", $p_remove_path="")
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::extractByIndex", "index='$p_index' path='$p_path', remove_path='$p_remove_path'");
    $v_result=1;

    // ----- Look if the $p_index is really an integer
    if (is_integer($p_index))
    {
      // ----- Call the extracting fct
      if (($v_result = $this->privExtractByIndex($p_list, "$p_index", $p_path, $p_remove_path)) != 1)
      {
        TrFctEnd(__FILE__, __LINE__, 0, PclErrorString());
        return(0);
      }
    }

    // ----- Look if the index is a string
    else if (is_string($p_index))
    {
      // ----- Call the extracting fct
      if (($v_result = $this->privExtractByIndex($p_list, $p_index, $p_path, $p_remove_path)) != 1)
      {
        TrFctEnd(__FILE__, __LINE__, 0, PclErrorString());
        return(0);
      }
    }

    // ----- Invalid variable
    else
    {
      // ----- Error log
      PclErrorLog(-3, "Invalid variable type $p_index");

      // ----- Return
      TrFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return 0;
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $p_list);
    return $p_list;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : deleteByIndex()
  // Description :
  //   This method delete somes files or folder entries from the zip archive.
  //   The entries (files or folders) are identified by their index in the
  //   archive (0-n).
  //   Note that if the index identify a folder entry, only the folder entry
  //   is deleted, and not all the files included in the folder.
  // Parameters :
  //   $p_index : A single index (integer) or a string of indexes of files to
  //              delete. The form of the string is "0,4-6,8-12" with only numbers
  //              and '-' for range or ',' to separate ranges. No spaces or ';'
  //              are allowed.
  // Return Values :
  //   0 on failure,
  //   The list of the files which are still present in the archive.
  //   (see PclZip::listContent() for list entry format)
  // --------------------------------------------------------------------------------
  function deleteByIndex($p_index)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::deleteByIndex", "index='$p_index'");
    $v_result=1;

    // ----- Look if the $p_index is really an integer
    if (is_integer($p_index))
    {
      // ----- Call the delete fct
      if (($v_result = $this->privDeleteByIndex("$p_index", $p_list)) != 1)
      {
        TrFctEnd(__FILE__, __LINE__, 0, PclErrorString());
        return(0);
      }
    }

    // ----- Look if the index is a string
    else if (is_string($p_index))
    {
      // ----- Call the delete fct
      if (($v_result = $this->privDeleteByIndex($p_index, $p_list)) != 1)
      {
        TrFctEnd(__FILE__, __LINE__, 0, PclErrorString());
        return(0);
      }
    }

    // ----- Invalid variable
    else
    {
      // ----- Error log
      PclErrorLog(-3, "Invalid variable type $p_index");

      // ----- Return
      TrFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return 0;
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $p_list);
    return $p_list;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : properties()
  // Description :
  //   This method gives the properties of the archive.
  //   The properties are :
  //     nb : Number of files in the archive
  //     comment : Comment associated with the archive file
  //     status : not_exist, ok
  // Parameters :
  //   None
  // Return Values :
  //   0 on failure,
  //   An array with the archive properties.
  // --------------------------------------------------------------------------------
  function properties()
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::properties", "");

    // ----- Default properties
    $v_prop = array();
    $v_prop['comment'] = '';
    $v_prop['nb'] = 0;
    $v_prop['status'] = 'not_exist';

    // ----- Look if file exists
    if (@is_file($this->zipname))
    {
      // ----- Open the zip file
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary read mode");
      if (($this->zip_fd = @fopen($this->zipname, 'rb')) == 0)
      {
        // ----- Error log
        PclErrorLog(-2, 'Unable to open archive \''.$this->zipname.'\' in binary read mode');

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), 0);
        return 0;
      }

      // ----- Read the central directory informations
      $v_central_dir = array();
      if (($v_result = $this->privReadEndCentralDir($v_central_dir)) != 1)
      {
        PclTraceFctEnd(__FILE__, __LINE__, 0);
        return 0;
      }

      // ----- Close the zip file
      $this->privCloseFd();

      // ----- Set the user attributes
      $v_prop['comment'] = $v_central_dir['comment'];
      $v_prop['nb'] = $v_central_dir['entries'];
      $v_prop['status'] = 'ok';
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_prop);
    return $v_prop;
  }
  // --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// ***** UNDER THIS LINE ARE DEFINED PRIVATE INTERNAL FUNCTIONS *****
// *****                                                        *****
// *****       THESES FUNCTIONS MUST NOT BE USED DIRECTLY       *****
// --------------------------------------------------------------------------------



  // --------------------------------------------------------------------------------
  // Function : privCreate()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privCreate($p_list, &$p_result_list, $p_add_dir="", $p_remove_dir="")
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privCreate", "list, result_list, add_dir='$p_add_dir', remove_dir='$p_remove_dir'");
    $v_result=1;
    $v_list_detail = array();

    // ----- Open the file in write mode
    if (($v_result = $this->privOpenFd('wb')) != 1)
    {
      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Add the list of files
    $v_result = $this->privAddList($p_list, $p_result_list, $p_add_dir, $p_remove_dir);

    // ----- Close
    $this->privCloseFd();

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privAdd()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privAdd($p_list, &$p_result_list, $p_add_dir="", $p_remove_dir="")
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privAdd", "list, result_list, add_dir='$p_add_dir', remove_dir='$p_remove_dir'");
    $v_result=1;
    $v_list_detail = array();

    // ----- Look if the archive exists
    if (!is_file($this->zipname))
    {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Archive does not exist, create it.");

      // ----- Do a create
      $v_result = $this->privCreate($p_list, $p_result_list, $p_add_dir, $p_remove_dir);

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Open the zip file
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary read mode");
    if (($v_result=$this->privOpenFd('rb')) != 1)
    {
      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Read the central directory informations
    $v_central_dir = array();
    if (($v_result = $this->privReadEndCentralDir($v_central_dir)) != 1)
    {
      $this->privCloseFd();
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Go to beginning of File
    PclTraceFctMessage(__FILE__, __LINE__, 5, "Position in file : ".ftell($this->zip_fd)."'");
    @rewind($this->zip_fd);
    PclTraceFctMessage(__FILE__, __LINE__, 5, "Position in file : ".ftell($this->zip_fd)."'");

    // ----- Creates a temporay file
    $v_zip_temp_name = PCLZIP_TEMPORARY_DIR.uniqid('pclzip-').'.tmp';

    // ----- Open the temporary file in write mode
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary read mode");
    if (($v_zip_temp_fd = @fopen($v_zip_temp_name, 'wb')) == 0)
    {
      $this->privCloseFd();

      PclErrorLog(-2, 'Unable to open temporary file \''.$v_zip_temp_name.'\' in binary write mode');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Copy the files from the archive to the temporary file
    // TBC : Here I should better append the file and go back to erase the central dir
    $v_size = $v_central_dir['offset'];
    while ($v_size != 0)
    {
      $v_read_size = ($v_size < PCLZIP_READ_BLOCK_SIZE ? $v_size : PCLZIP_READ_BLOCK_SIZE);
      PclTraceFctMessage(__FILE__, __LINE__, 4, "Read $v_read_size bytes");
      $v_buffer = fread($this->zip_fd, $v_read_size);
      @fwrite($v_zip_temp_fd, $v_buffer, $v_read_size);
      $v_size -= $v_read_size;
    }

    // ----- Swap the file descriptor
    // Here is a trick : I swap the temporary fd with the zip fd, in order to use
    // the following methods on the temporary fil and not the real archive
    $v_swap = $this->zip_fd;
    $this->zip_fd = $v_zip_temp_fd;
    $v_zip_temp_fd = $v_swap;

    // ----- Add the files
    $v_header_list = array();
    if (($v_result = $this->privAddFileList($p_list, $v_header_list, $p_add_dir, $p_remove_dir)) != 1)
    {
      fclose($v_zip_temp_fd);
      $this->privCloseFd();
      @unlink($v_zip_temp_name);

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Store the offset of the central dir
    $v_offset = @ftell($this->zip_fd);
    PclTraceFctMessage(__FILE__, __LINE__, 4, "New offset of central dir : $v_offset");

    // ----- Copy the block of file headers from the old archive
    $v_size = $v_central_dir['size'];
    while ($v_size != 0)
    {
      $v_read_size = ($v_size < PCLZIP_READ_BLOCK_SIZE ? $v_size : PCLZIP_READ_BLOCK_SIZE);
      PclTraceFctMessage(__FILE__, __LINE__, 4, "Read $v_read_size bytes");
      $v_buffer = @fread($v_zip_temp_fd, $v_read_size);
      @fwrite($this->zip_fd, $v_buffer, $v_read_size);
      $v_size -= $v_read_size;
    }

    // ----- Create the Central Dir files header
    for ($i=0; $i<sizeof($v_header_list); $i++)
    {
      // ----- Create the file header
      if (($v_result = $this->privWriteCentralFileHeader($v_header_list[$i])) != 1)
      {
        fclose($v_zip_temp_fd);
        $this->privCloseFd();
        @unlink($v_zip_temp_name);

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Transform the header to a 'usable' info
      $this->privConvertHeader2FileInfo($v_header_list[$i], $p_result_list[$i]);
    }

    // ----- Zip file comment
    $v_comment = '';

    // ----- Calculate the size of the central header
    $v_size = @ftell($this->zip_fd)-$v_offset;

    // ----- Create the central dir footer
    if (($v_result = $this->privWriteCentralHeader(sizeof($v_header_list)+$v_central_dir['entries'], $v_size, $v_offset, $v_comment)) != 1)
    {
      // ----- Reset the file list
      unset($v_header_list);

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Swap back the file descriptor
    $v_swap = $this->zip_fd;
    $this->zip_fd = $v_zip_temp_fd;
    $v_zip_temp_fd = $v_swap;

    // ----- Close
    $this->privCloseFd();

    // ----- Close the temporary file
    @fclose($v_zip_temp_fd);

    // ----- Delete the zip file
    // TBC : I should test the result ...
    @unlink($this->zipname);

    // ----- Rename the temporary file
    // TBC : I should test the result ...
    //@rename($v_zip_temp_name, $this->zipname);
    PclZipUtilRename($v_zip_temp_name, $this->zipname);

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privOpenFd()
  // Description :
  // Parameters :
  // --------------------------------------------------------------------------------
  function privOpenFd($p_mode)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privOpenFd", 'mode='.$p_mode);
    $v_result=1;

    // ----- Look if already open
    if ($this->zip_fd != 0)
    {
      // ----- Error log
      PclErrorLog(-2, 'Zip file \''.$this->zipname.'\' already open');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Open the zip file
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Open file in '.$p_mode.' mode');
    if (($this->zip_fd = @fopen($this->zipname, $p_mode)) == 0)
    {
      // ----- Error log
      PclErrorLog(-2, 'Unable to open archive \''.$this->zipname.'\' in '.$p_mode.' mode');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privCloseFd()
  // Description :
  // Parameters :
  // --------------------------------------------------------------------------------
  function privCloseFd()
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privCloseFd", "");
    $v_result=1;

    if ($this->zip_fd != 0)
      @fclose($this->zip_fd);
    $this->zip_fd = 0;

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privAddList()
  // Description :
  //   $p_add_dir and $p_remove_dir will give the ability to memorize a path which is
  //   different from the real path of the file. This is usefull if you want to have PclTar
  //   running in any directory, and memorize relative path from an other directory.
  // Parameters :
  //   $p_list : An array containing the file or directory names to add in the tar
  //   $p_result_list : list of added files with their properties (specially the status field)
  //   $p_add_dir : Path to add in the filename path archived
  //   $p_remove_dir : Path to remove in the filename path archived
  // Return Values :
  // --------------------------------------------------------------------------------
  function privAddList($p_list, &$p_result_list, $p_add_dir, $p_remove_dir)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privAddList", "list, add_dir='$p_add_dir', remove_dir='$p_remove_dir'");
    $v_result=1;

    // ----- Add the files
    $v_header_list = array();
    if (($v_result = $this->privAddFileList($p_list, $v_header_list, $p_add_dir, $p_remove_dir)) != 1)
    {
      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Store the offset of the central dir
    $v_offset = @ftell($this->zip_fd);

    // ----- Create the Central Dir files header
    for ($i=0; $i<sizeof($v_header_list); $i++)
    {
      // ----- Create the file header
      if (($v_result = $this->privWriteCentralFileHeader($v_header_list[$i])) != 1)
      {
        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Transform the header to a 'usable' info
      $this->privConvertHeader2FileInfo($v_header_list[$i], $p_result_list[$i]);
    }

    // ----- Zip file comment
    $v_comment = '';

    // ----- Calculate the size of the central header
    $v_size = @ftell($this->zip_fd)-$v_offset;

    // ----- Create the central dir footer
    if (($v_result = $this->privWriteCentralHeader(sizeof($v_header_list), $v_size, $v_offset, $v_comment)) != 1)
    {
      // ----- Reset the file list
      unset($v_header_list);

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privAddFileList()
  // Description :
  //   $p_add_dir and $p_remove_dir will give the ability to memorize a path which is
  //   different from the real path of the file. This is usefull if you want to
  //   run the lib in any directory, and memorize relative path from an other directory.
  // Parameters :
  //   $p_list : An array containing the file or directory names to add in the tar
  //   $p_result_list : list of added files with their properties (specially the status field)
  //   $p_add_dir : Path to add in the filename path archived
  //   $p_remove_dir : Path to remove in the filename path archived
  // Return Values :
  // --------------------------------------------------------------------------------
  function privAddFileList($p_list, &$p_result_list, $p_add_dir, $p_remove_dir)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privAddFileList", "list, add_dir='$p_add_dir', remove_dir='$p_remove_dir'");
    $v_result=1;
    $v_header = array();

    // ----- Recuperate the current number of elt in list
    $v_nb = sizeof($p_result_list);
    PclTraceFctMessage(__FILE__, __LINE__, 4, "Before add, list have $v_nb elements");

    // ----- Loop on the files
    for ($j=0; ($j<count($p_list)) && ($v_result==1); $j++)
    {
      // ----- Recuperate the filename
      $p_filename = $p_list[$j];

      PclTraceFctMessage(__FILE__, __LINE__, 2, "Looking for file [$p_filename]");

      // ----- Skip empty file names
      if ($p_filename == "")
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Skip empty filename");
        continue;
      }

      // ----- Check the filename
      if (!file_exists($p_filename))
      {
        // ----- Error log
        PclTraceFctMessage(__FILE__, __LINE__, 2, "File '$p_filename' does not exists");
        PclErrorLog(-4, "File '$p_filename' does not exists");

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }

      // ----- Check the path length
      if (strlen($p_filename) > 0xFF)
      {
        // ----- Error log
        PclErrorLog(-5, "File name is too long (max. 255) : '$p_filename'");

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }

      // ----- Add the file
      if (($v_result = $this->privAddFile($p_filename, $v_header, $p_add_dir, $p_remove_dir)) != 1)
      {
        // ----- Return status
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Store the file infos
      $p_result_list[$v_nb++] = $v_header;

      // ----- Look for directory
      if (is_dir($p_filename))
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "$p_filename is a directory");

        // ----- Look for path
        if ($p_filename != ".")
          $v_path = $p_filename."/";
        else
          $v_path = "";

        // ----- Read the directory for files and sub-directories
        $p_hdir = opendir($p_filename);
        $p_hitem = readdir($p_hdir); // '.' directory
        $p_hitem = readdir($p_hdir); // '..' directory
        while ($p_hitem = readdir($p_hdir))
        {
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Looking for $p_hitem in the directory");

          // ----- Look for a file
          if (is_file($v_path.$p_hitem))
          {
            PclTraceFctMessage(__FILE__, __LINE__, 4, "Add the file '".$v_path.$p_hitem."'");

            // ----- Add the file
            if (($v_result = $this->privAddFile($v_path.$p_hitem, $v_header, $p_add_dir, $p_remove_dir)) != 1)
            {
              // ----- Return status
              PclTraceFctEnd(__FILE__, __LINE__, $v_result);
              return $v_result;
            }

            // ----- Store the file infos
            $p_result_list[$v_nb++] = $v_header;
          }

          // ----- Recursive call to PclTarHandleAddFile()
          else
          {
            PclTraceFctMessage(__FILE__, __LINE__, 4, "Add the directory '".$v_path.$p_hitem."'");

            // ----- Need an array as parameter
            $p_temp_list[0] = $v_path.$p_hitem;
            $v_result = $this->privAddFileList($p_temp_list, $p_result_list, $p_add_dir, $p_remove_dir);

            // ----- Update the number of elements of the list
            $v_nb = sizeof($p_result_list);
          }
        }

        // ----- Free memory for the recursive loop
        unset($p_temp_list);
        unset($p_hdir);
        unset($p_hitem);
      }
/*
      // ----- Add the file
      if (($v_result = $this->privAddFile($p_filename, $v_header, $p_add_dir, $p_remove_dir)) != 1)
      {
        // ----- Return status
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Store the file infos
      $p_result_list[$v_nb++] = $v_header;
*/
    }

    PclTraceFctMessage(__FILE__, __LINE__, 4, "After add, list have $v_nb elements");

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privAddFile()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privAddFile($p_filename, &$p_header, $p_add_dir, $p_remove_dir)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privAddFile", "filename='$p_filename', add_dir='$p_add_dir', remove_dir='$p_remove_dir'");
    $v_result=1;

    if ($p_filename == "")
    {
      // ----- Error log
      PclErrorLog(-3, "Invalid file list parameter (invalid or empty list)");

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Calculate the stored filename
    $v_stored_filename = $p_filename;
    if ($p_remove_dir != "")
    {
      if (substr($p_remove_dir, -1) != '/')
        $p_remove_dir .= "/";

      if ((substr($p_filename, 0, 2) == "./") || (substr($p_remove_dir, 0, 2) == "./"))
      {
        if ((substr($p_filename, 0, 2) == "./") && (substr($p_remove_dir, 0, 2) != "./"))
          $p_remove_dir = "./".$p_remove_dir;
        if ((substr($p_filename, 0, 2) != "./") && (substr($p_remove_dir, 0, 2) == "./"))
          $p_remove_dir = substr($p_remove_dir, 2);
      }

      if (substr($p_filename, 0, strlen($p_remove_dir)) == $p_remove_dir)
      {
        $v_stored_filename = substr($p_filename, strlen($p_remove_dir));
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Remove path '$p_remove_dir' in file '$p_filename' = '$v_stored_filename'");
      }
    }
    if ($p_add_dir != "")
    {
      if (substr($p_add_dir, -1) == "/")
        $v_stored_filename = $p_add_dir.$v_stored_filename;
      else
        $v_stored_filename = $p_add_dir."/".$v_stored_filename;
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Add path '$p_add_dir' in file '$p_filename' = '$v_stored_filename'");
    }

    // ----- Filename (reduce the path of stored name)
    $v_stored_filename = PclZipUtilPathReduction($v_stored_filename);
    PclTraceFctMessage(__FILE__, __LINE__, 2, "Filename (reduced) '$v_stored_filename', strlen ".strlen($v_stored_filename));

    // ----- Check the path length
    if (strlen($v_stored_filename) > 0xFF)
    {
      // ----- Error log
      PclErrorLog(-5, "Stored file name is too long (max. 255) : '$v_stored_filename'");

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Set the file properties
    clearstatcache();
    $p_header['version'] = 20;
    $p_header['version_extracted'] = 10;
    $p_header['flag'] = 0;
    $p_header['compression'] = 0;
    $p_header['mtime'] = filemtime($p_filename);
    $p_header['crc'] = 0;
    $p_header['compressed_size'] = 0;
    $p_header['size'] = filesize($p_filename);
    $p_header['filename_len'] = strlen($p_filename);
    $p_header['extra_len'] = 0;
    $p_header['comment_len'] = 0;
    $p_header['disk'] = 0;
    $p_header['internal'] = 0;
    $p_header['external'] = 0xFE49FFE0;    // Value for a file : To be checked
    $p_header['offset'] = 0;
    $p_header['filename'] = $p_filename;
    $p_header['stored_filename'] = $v_stored_filename;
    $p_header['extra'] = '';
    $p_header['comment'] = '';
    $p_header['status'] = 'ok';

    // ----- Look for a file
    if (is_file($p_filename))
    {
      // ----- Open the source file
      if (($v_file = @fopen($p_filename, "rb")) == 0)
      {
        // ----- Error log
        PclErrorLog(-2, "Unable to open file '$p_filename' in binary read mode");

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }

      // ----- Creates a compressed temporary file
      if (($v_file_compressed = @gzopen($p_filename.'.gz', "wb")) == 0)
      {
        // ----- Close the file
        fclose($v_file);

        // ----- Error log
        PclErrorLog(-1, "Unable to open file '$p_filename.gz' in gz binary write mode");

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }

      // ----- Read the file by PCLZIP_READ_BLOCK_SIZE octets blocks
      $v_size = filesize($p_filename);
      while ($v_size != 0)
      {
        $v_read_size = ($v_size < PCLZIP_READ_BLOCK_SIZE ? $v_size : PCLZIP_READ_BLOCK_SIZE);
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Read $v_read_size bytes");
        $v_buffer = fread($v_file, $v_read_size);
        $v_binary_data = pack('a'.$v_read_size, $v_buffer);
        @gzputs($v_file_compressed, $v_binary_data, $v_read_size);
        $v_size -= $v_read_size;
      }

      // ----- Close the file
      @fclose($v_file);
      @gzclose($v_file_compressed);

      // ----- Check the minimum file size
      PclTraceFctMessage(__FILE__, __LINE__, 4, "gzip file size ".filesize($p_filename.'.gz'));
      if (filesize($p_filename.'.gz') < 18)
      {
        // ----- Error log
        PclErrorLog(-2, 'Invalid file "'.$p_filename.'.gz'.'" size (less than header size)');

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }

      // ----- Extract the compressed attributes
      if (($v_file_compressed = @fopen($p_filename.'.gz', "rb")) == 0)
      {
        // ----- Error log
        PclErrorLog(-2, "Unable to open file '$p_filename.gz' in gz binary read mode");

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }

      // ----- Read the gzip file header
      $v_binary_data = @fread($v_file_compressed, 10);
      $v_data_header = unpack('a1id1/a1id2/a1cm/a1flag/Vmtime/a1xfl/a1os', $v_binary_data);

      // ----- Check some parameters
      PclTraceFctMessage(__FILE__, __LINE__, 4, '$v_data_header[id1]='.bin2hex($v_data_header['id1']));
      PclTraceFctMessage(__FILE__, __LINE__, 4, '$v_data_header[id2]='.bin2hex($v_data_header['id2']));
      PclTraceFctMessage(__FILE__, __LINE__, 4, '$v_data_header[cm]='.bin2hex($v_data_header['cm']));
      PclTraceFctMessage(__FILE__, __LINE__, 4, '$v_data_header[flag]='.bin2hex($v_data_header['flag']));
      PclTraceFctMessage(__FILE__, __LINE__, 4, '$v_data_header[mtime]='.$v_data_header['mtime']);
      PclTraceFctMessage(__FILE__, __LINE__, 4, '$v_data_header[xfl]='.bin2hex($v_data_header['xfl']));
      PclTraceFctMessage(__FILE__, __LINE__, 4, '$v_data_header[os]='.bin2hex($v_data_header['os']));
      $v_data_header['os'] = bin2hex($v_data_header['os']);

      // ----- Read the gzip file footer
      PclTraceFctMessage(__FILE__, __LINE__, 4, "File position after header ".ftell($v_file_compressed));
      @fseek($v_file_compressed, filesize($p_filename.'.gz')-8);
      PclTraceFctMessage(__FILE__, __LINE__, 4, "File position at beginning of footer ".ftell($v_file_compressed));
      $v_binary_data = @fread($v_file_compressed, 8);
      PclTraceFctMessage(__FILE__, __LINE__, 4, "File position after footer ".ftell($v_file_compressed));
      $v_data_footer = unpack('Vcrc/Vcompressed_size', $v_binary_data);

      // ----- Set the attributes
      $p_header['compression'] = ord($v_data_header['cm']);
      //$p_header['mtime'] = $v_data_header['mtime'];
      $p_header['crc'] = $v_data_footer['crc'];
      PclTraceFctMessage(__FILE__, __LINE__, 4, "Compressed size ".(filesize($p_filename.'.gz')-18));
      $p_header['compressed_size'] = filesize($p_filename.'.gz')-18;

      // ----- Close the file
      @fclose($v_file_compressed);

      // ----- Call the header generation
      if (($v_result = $this->privWriteFileHeader($p_header)) != 1)
      {
        // ----- Return status
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Add the compressed data
      if (($v_file_compressed = @fopen($p_filename.'.gz', "rb")) == 0)
      {
        // ----- Error log
        PclErrorLog(-2, "Unable to open file '$p_filename.gz' in gz binary read mode");

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }

      // ----- Read the file by PCLZIP_READ_BLOCK_SIZE octets blocks
      fseek($v_file_compressed, 10);
      PclTraceFctMessage(__FILE__, __LINE__, 4, "File position before reading compressed data ".ftell($v_file_compressed));
      PclTraceFctMessage(__FILE__, __LINE__, 4, ' '.$p_header['compressed_size'].' bytes to read');
      $v_size = $p_header['compressed_size'];
      while ($v_size != 0)
      {
        $v_read_size = ($v_size < PCLZIP_READ_BLOCK_SIZE ? $v_size : PCLZIP_READ_BLOCK_SIZE);
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Read $v_read_size bytes");
        $v_buffer = fread($v_file_compressed, $v_read_size);
        $v_binary_data = pack('a'.$v_read_size, $v_buffer);
        @fwrite($this->zip_fd, $v_binary_data, $v_read_size);
        $v_size -= $v_read_size;
      }

      // ----- Close the file
      @fclose($v_file_compressed);

      // ----- Unlink the temporary file
      @unlink($p_filename.'.gz');
    }

    // ----- Look for a directory
    else
    {
      // ----- Set the file properties
      $p_header['filename'] .= '/';
      $p_header['filename_len']++;
      $p_header['size'] = 0;
      $p_header['external'] = 0x41FF0010;   // Value for a folder : to be checked

      // ----- Call the header generation
      if (($v_result = $this->privWriteFileHeader($p_header)) != 1)
      {
        // ----- Return status
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privWriteFileHeader()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privWriteFileHeader(&$p_header)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privWriteFileHeader", 'file="'.$p_header['filename'].'", stored as "'.$p_header['stored_filename'].'"');
    $v_result=1;

    // TBC
    for(reset($p_header); $key = key($p_header); next($p_header)) {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "header[$key] = ".$p_header[$key]);
    }

    // ----- Store the offset position of the file
    $p_header['offset'] = ftell($this->zip_fd);
    PclTraceFctMessage(__FILE__, __LINE__, 2, 'File offset of the header :'.$p_header['offset']);

    // ----- Transform UNIX mtime to DOS format mdate/mtime
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Date : \''.date("d/m/y H:i:s", $p_header['mtime']).'\'');
    $v_date = getdate($p_header['mtime']);
    $v_mtime = ($v_date['hours']<<11) + ($v_date['minutes']<<5) + $v_date['seconds']/2;
    $v_mdate = (($v_date['year']-1980)<<9) + ($v_date['mon']<<5) + $v_date['mday'];

    // ----- Packed data
    $v_binary_data = pack("VvvvvvVVVvv", 0x04034b50, $p_header['version'], $p_header['flag'],
                          $p_header['compression'], $v_mtime, $v_mdate,
                          $p_header['crc'], $p_header['compressed_size'], $p_header['size'],
                          strlen($p_header['stored_filename']), $p_header['extra_len']);

    // ----- Write the first 148 bytes of the header in the archive
    fputs($this->zip_fd, $v_binary_data, 30);

    // ----- Write the variable fields
    if (strlen($p_header['stored_filename']) != 0)
    {
      fputs($this->zip_fd, $p_header['stored_filename'], strlen($p_header['stored_filename']));
    }
    if ($p_header['extra_len'] != 0)
    {
      fputs($this->zip_fd, $p_header['extra'], $p_header['extra_len']);
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privWriteCentralFileHeader()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privWriteCentralFileHeader(&$p_header)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privWriteCentralFileHeader", 'file="'.$p_header['filename'].'", stored as "'.$p_header['stored_filename'].'"');
    $v_result=1;

    // TBC
    for(reset($p_header); $key = key($p_header); next($p_header)) {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "header[$key] = ".$p_header[$key]);
     }

    // ----- Transform UNIX mtime to DOS format mdate/mtime
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Date : \''.date("d/m/y H:i:s", $p_header['mtime']).'\'');
    $v_date = getdate($p_header['mtime']);
    $v_mtime = ($v_date['hours']<<11) + ($v_date['minutes']<<5) + $v_date['seconds']/2;
    $v_mdate = (($v_date['year']-1980)<<9) + ($v_date['mon']<<5) + $v_date['mday'];

    // ----- Packed data
    $v_binary_data = pack("VvvvvvvVVVvvvvvVV", 0x02014b50, $p_header['version'], $p_header['version_extracted'],
                          $p_header['flag'], $p_header['compression'], $v_mtime, $v_mdate, $p_header['crc'],
                          $p_header['compressed_size'], $p_header['size'],
                          strlen($p_header['stored_filename']), $p_header['extra_len'], $p_header['comment_len'],
                          $p_header['disk'], $p_header['internal'], $p_header['external'], $p_header['offset']);

    // ----- Write the 42 bytes of the header in the zip file
    fputs($this->zip_fd, $v_binary_data, 46);

    // ----- Write the variable fields
    if (strlen($p_header['stored_filename']) != 0)
    {
      fputs($this->zip_fd, $p_header['stored_filename'], strlen($p_header['stored_filename']));
    }
    if ($p_header['extra_len'] != 0)
    {
      fputs($this->zip_fd, $p_header['extra'], $p_header['extra_len']);
    }
    if ($p_header['comment_len'] != 0)
    {
      fputs($this->zip_fd, $p_header['comment'], $p_header['comment_len']);
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privWriteCentralHeader()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privWriteCentralHeader($p_nb_entries, $p_size, $p_offset, $p_comment)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privWriteCentralHeader", 'nb_entries='.$p_nb_entries.', size='.$p_size.', offset='.$p_offset.', comment="'.$p_comment.'"');
    $v_result=1;

    // ----- Packed data
    $v_binary_data = pack("VvvvvVVv", 0x06054b50, 0, 0, $p_nb_entries, $p_nb_entries, $p_size, $p_offset, strlen($p_comment));

    // ----- Write the 22 bytes of the header in the zip file
    fputs($this->zip_fd, $v_binary_data, 22);

    // ----- Write the variable fields
    if (strlen($p_comment) != 0)
    {
      fputs($this->zip_fd, $p_comment, strlen($p_comment));
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privList()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privList(&$p_list)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privList", "list");
    $v_result=1;

    // ----- Open the zip file
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary read mode");
    if (($this->zip_fd = @fopen($this->zipname, 'rb')) == 0)
    {
      // ----- Error log
      PclErrorLog(-2, 'Unable to open archive \''.$this->zipname.'\' in binary read mode');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Read the central directory informations
    $v_central_dir = array();
    if (($v_result = $this->privReadEndCentralDir($v_central_dir)) != 1)
    {
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Go to beginning of Central Dir
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Offset : ".$v_central_dir['offset']."'");
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Position in file : ".ftell($this->zip_fd)."'");
    @rewind($this->zip_fd);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Position in file : ".ftell($this->zip_fd)."'");
    if (@fseek($this->zip_fd, $v_central_dir['offset']))
    {
      // ----- Error log
      PclErrorLog(-14, 'Invalid archive size');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Position in file : ".ftell($this->zip_fd)."'");

    // ----- Read each entry
    for ($i=0; $i<$v_central_dir['entries']; $i++)
    {
      // ----- Read the file header
      if (($v_result = $this->privReadCentralFileHeader(&$v_header)) != 1)
      {
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
      $v_header['index'] = $i;

      // ----- Get the only interesting attributes
      $this->privConvertHeader2FileInfo($v_header, $p_list[$i]);
      unset($v_header);
    }

    // ----- Close the zip file
    $this->privCloseFd();

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privConvertHeader2FileInfo()
  // Description :
  //   This function takes the file informations from the central directory
  //   entries and extract the interesting parameters that will be given back.
  //   The resulting file infos are set in the array $p_info
  //     $p_info['filename'] : Filename with full path. Given by user (add),
  //                           extracted in the filesystem (extract).
  //     $p_info['stored_filename'] : Stored filename in the archive.
  //     $p_info['size'] = Size of the file.
  //     $p_info['compressed_size'] = Compressed size of the file.
  //     $p_info['mtime'] = Last modification date of the file.
  //     $p_info['comment'] = Comment associated with the file.
  //     $p_info['folder'] = true/false : indicates if the entry is a folder or not.
  //     $p_info['status'] = status of the action on the file.
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privConvertHeader2FileInfo($p_header, &$p_info)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privConvertHeader2FileInfo", "Filename='".$p_header['filename']."'");
    $v_result=1;

    // ----- Get the interesting attributes
    $p_info['filename'] = $p_header['filename'];
    $p_info['stored_filename'] = $p_header['stored_filename'];
    $p_info['size'] = $p_header['size'];
    $p_info['compressed_size'] = $p_header['compressed_size'];
    $p_info['mtime'] = $p_header['mtime'];
    $p_info['comment'] = $p_header['comment'];
    $p_info['folder'] = ($p_header['external']==0x41FF0010);
    $p_info['index'] = $p_header['index'];
    $p_info['status'] = $p_header['status'];
    // TBC
    /*
    for(reset($p_header); $key = key($p_header); next($p_header)) {
      $p_info[$key] = $p_header[$key];
    }
    */

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privExtract()
  // Description :
  // Parameters :
  //   $p_file_list : An array where will be placed the properties of each
  //                  extracted file
  //   $p_path : Path to add while writing the extracted files
  //   $p_remove_path : Path to remove (from the file memorized path) while writing the
  //                    extracted files. If the path does not match the file path,
  //                    the file is extracted with its memorized path.
  //                    $p_remove_path does not apply to 'list' mode.
  //                    $p_path and $p_remove_path are commulative.
  // Return Values :
  //   1 on success,0 or less on error (see error code list)
  // --------------------------------------------------------------------------------
  function privExtract(&$p_file_list, $p_path, $p_remove_path)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privExtract", "list, path=$p_path, remove_path='$p_remove_path'");
    $v_result=1;

    // ----- Check the path
    if (($p_path == "") || ((substr($p_path, 0, 1) != "/") && (substr($p_path, 0, 3) != "../")))
      $p_path = "./".$p_path;

    // ----- Reduce the path last (and duplicated) '/'
    if (($p_path != "./") && ($p_path != "/"))
    {
      // ----- Look for the path end '/'
      while (substr($p_path, -1) == "/")
      {
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Destination path [$p_path] ends by '/'");
        $p_path = substr($p_path, 0, strlen($p_path)-1);
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Modified to [$p_path]");
      }
    }

    // ----- Look for path to remove format (should end by /)
    if (($p_remove_path != "") && (substr($p_remove_path, -1) != '/'))
    {
      $p_remove_path .= '/';
    }
    $p_remove_path_size = strlen($p_remove_path);

    // ----- Open the zip file
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary read mode");
    if (($v_result = $this->privOpenFd('rb')) != 1)
    {
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Read the central directory informations
    $v_central_dir = array();
    if (($v_result = $this->privReadEndCentralDir($v_central_dir)) != 1)
    {
      // ----- Close the zip file
      $this->privCloseFd();

      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Start at beginning of Central Dir
    $v_pos_entry = $v_central_dir['offset'];

    // ----- Read each entry
    for ($i=0; $i<$v_central_dir['entries']; $i++)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Read next file entry : $i'");

      // ----- Read next Central dir entry
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position before rewind : ".ftell($this->zip_fd)."'");
      @rewind($this->zip_fd);
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after rewind : ".ftell($this->zip_fd)."'");
      if (@fseek($this->zip_fd, $v_pos_entry))
      {
        // ----- Close the zip file
        $this->privCloseFd();

        // ----- Error log
        PclErrorLog(-14, 'Invalid archive size');

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after fseek : ".ftell($this->zip_fd)."'");

      // ----- Read the file header
      $v_header = array();
      if (($v_result = $this->privReadCentralFileHeader(&$v_header)) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();

        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Store the index
      $v_header['index'] = $i;

      // ----- Store the file position
      $v_pos_entry = ftell($this->zip_fd);

      // ----- Go to the file position
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position before rewind : ".ftell($this->zip_fd)."'");
      @rewind($this->zip_fd);
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after rewind : ".ftell($this->zip_fd)."'");
      if (@fseek($this->zip_fd, $v_header['offset']))
      {
        // ----- Close the zip file
        $this->privCloseFd();

        // ----- Error log
        PclErrorLog(-14, 'Invalid archive size');

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after fseek : ".ftell($this->zip_fd)."'");

      // ----- Extracting the file
      if (($v_result = $this->privExtractFile($v_header, $p_path, $p_remove_path)) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();

        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Get the only interesting attributes
      if (($v_result = $this->privConvertHeader2FileInfo($v_header, $p_file_list[$i])) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();

        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
    }

    // ----- Close the zip file
    $this->privCloseFd();

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privExtractByIndex()
  // Description :
  // Parameters :
  //   $p_file_list : An array where will be placed the properties of each
  //                  extracted file
  //   $p_index : A single index (integer) or a string of indexes of files to
  //              extract. The form of the string is "0,4-6,8-12" with only numbers
  //              and '-' for range or ',' to separate ranges. No spaces or ';'
  //              are allowed.
  //   $p_path : Path to add while writing the extracted files
  //   $p_remove_path : Path to remove (from the file memorized path) while writing the
  //                    extracted files. If the path does not match the file path,
  //                    the file is extracted with its memorized path.
  //                    $p_remove_path does not apply to 'list' mode.
  //                    $p_path and $p_remove_path are commulative.
  // Return Values :
  //   1 on success,0 or less on error (see error code list)
  // --------------------------------------------------------------------------------
  function privExtractByIndex(&$p_file_list, $p_index, $p_path, $p_remove_path)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privExtractByIndex", "list, index='$p_index', path=$p_path, remove_path='$p_remove_path'");
    $v_result=1;

    // ----- Check the path
    if (($p_path == "") || ((substr($p_path, 0, 1) != "/") && (substr($p_path, 0, 3) != "../")))
      $p_path = "./".$p_path;

    // ----- Reduce the path last (and duplicated) '/'
    if (($p_path != "./") && ($p_path != "/"))
    {
      // ----- Look for the path end '/'
      while (substr($p_path, -1) == "/")
      {
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Destination path [$p_path] ends by '/'");
        $p_path = substr($p_path, 0, strlen($p_path)-1);
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Modified to [$p_path]");
      }
    }

    // ----- Look for path to remove format (should end by /)
    if (($p_remove_path != "") && (substr($p_remove_path, -1) != '/'))
    {
      $p_remove_path .= '/';
    }
    $p_remove_path_size = strlen($p_remove_path);

    // ----- Open the zip file
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary read mode");
    if (($v_result = $this->privOpenFd('rb')) != 1)
    {
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Read the central directory informations
    $v_central_dir = array();
    if (($v_result = $this->privReadEndCentralDir($v_central_dir)) != 1)
    {
      // ----- Close the zip file
      $this->privCloseFd();

      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Manipulate the index list
    // TBC : In PHP4, we should have a char replace better than a string replace
    $p_index = str_replace(' ', '', $p_index);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Reduced index : '$p_index'");
    $v_index_list = explode(",", $p_index);
    sort($v_index_list);

    // ----- Start at beginning of Central Dir
    $v_pos_entry = $v_central_dir['offset'];

    // ----- Read each entry
    for ($i=0, $j_start=0, $v_nb_extracted=0; ($i<$v_central_dir['entries']) && ($j_start<sizeof($v_index_list)); $i++)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Read next file header entry : $i'");

      // ----- Read next Central dir entry
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position before rewind : ".ftell($this->zip_fd)."'");
      @rewind($this->zip_fd);
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after rewind : ".ftell($this->zip_fd)."'");
      if (@fseek($this->zip_fd, $v_pos_entry))
      {
        // ----- Close the zip file
        $this->privCloseFd();

        // ----- Error log
        PclErrorLog(-14, 'Invalid archive size');

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after fseek : ".ftell($this->zip_fd)."'");

      // ----- Read the file header
      $v_header = array();
      if (($v_result = $this->privReadCentralFileHeader(&$v_header)) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();

        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Store the index
      $v_header['index'] = $i;

      // ----- Store the file position
      $v_pos_entry = ftell($this->zip_fd);

      // ----- Look if the index is in the list
      $v_extract = false;
      for ($j=$j_start; ($j<sizeof($v_index_list)) && (!$v_extract); $j++)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Look if index '$i' is in '".$v_index_list[$j]."'");

        // ----- Extract range
        $v_item_list = explode("-", $v_index_list[$j]);
        $v_size_item_list = sizeof($v_item_list);
        if ($v_size_item_list == 1)
        {
          if ($i==$v_item_list[0])
          {
            PclTraceFctMessage(__FILE__, __LINE__, 3, "Found as a single index");
            $v_extract = true;
          }
          if ($i>=$v_item_list[0])
          {
            $j_start = $j+1;
          }
        }
        else if ($v_size_item_list == 2)
        {
          if (($i>=$v_item_list[0]) && ($i<=$v_item_list[1]))
          {
            PclTraceFctMessage(__FILE__, __LINE__, 3, "Found as part of an index range");
            $v_extract = true;
          }
          if ($i>=$v_item_list[1])
          {
            $j_start = $j+1;
          }
        }
        if ($v_item_list[0]>$i)
        {
          PclTraceFctMessage(__FILE__, __LINE__, 3, "Range is greater than index, stop loop");
          break;
        }
      }

      // ----- Look for real extraction
      if ($v_extract)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Extracting file '".$v_header['filename']."', index '$i'");

        // ----- Go to the file position
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Position before rewind : ".ftell($this->zip_fd)."'");
        @rewind($this->zip_fd);
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after rewind : ".ftell($this->zip_fd)."'");
        if (@fseek($this->zip_fd, $v_header['offset']))
        {
          // ----- Close the zip file
          $this->privCloseFd();

          // ----- Error log
          PclErrorLog(-14, 'Invalid archive size');

          // ----- Return
          PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
          return PclErrorCode();
        }
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after fseek : ".ftell($this->zip_fd)."'");

        // ----- Extracting the file
        if (($v_result = $this->privExtractFile($v_header, $p_path, $p_remove_path)) != 1)
        {
          // ----- Close the zip file
          $this->privCloseFd();

          PclTraceFctEnd(__FILE__, __LINE__, $v_result);
          return $v_result;
        }

        // ----- Get the only interesting attributes
        if (($v_result = $this->privConvertHeader2FileInfo($v_header, $p_file_list[$v_nb_extracted++])) != 1)
        {
          // ----- Close the zip file
          $this->privCloseFd();

          PclTraceFctEnd(__FILE__, __LINE__, $v_result);
          return $v_result;
        }
      }
    }

    // ----- Close the zip file
    $this->privCloseFd();

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privExtractFile()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privExtractFile(&$p_entry, $p_path, $p_remove_path)
  {
    PclTraceFctStart(__FILE__, __LINE__, 'PclZip::privExtractFile', "path='$p_path', remove_path='$p_remove_path'");
    $v_result=1;

    // ----- Read the file header
    if (($v_result = $this->privReadFileHeader(&$v_header)) != 1)
    {
      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    PclTraceFctMessage(__FILE__, __LINE__, 2, "Found file '".$v_header['filename']."', size '".$v_header['size']."'");

    // ----- Check that the file header is coherent with $p_entry info
    // TBC

    // ----- Look for path to remove
    if ($p_remove_path != "")
    {
      if (strcmp($p_remove_path, $p_entry['filename'])==0)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "The folder is the same as the removed path '".$p_entry['filename']."'");

        // ----- Change the file status
        $p_entry['status'] = "filtered";

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      $p_remove_path_size = strlen($p_remove_path);
      if (substr($p_entry['filename'], 0, $p_remove_path_size) == $p_remove_path)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Found path '$p_remove_path' to remove in file '".$p_entry['filename']."'");

        // ----- Remove the path
        $p_entry['filename'] = substr($p_entry['filename'], $p_remove_path_size);

        PclTraceFctMessage(__FILE__, __LINE__, 3, "Resulting file is '".$p_entry['filename']."'");
      }
    }

    // ----- Add the path
    if ($p_path != '')
    {
      $p_entry['filename'] = $p_path."/".$p_entry['filename'];
    }

    // ----- Trace
    PclTraceFctMessage(__FILE__, __LINE__, 2, "Extracting file (with path) '".$p_entry['filename']."', size '$v_header[size]'");

    // ----- Check that the file does not exists
    if (file_exists($p_entry['filename']))
    {
      PclTraceFctMessage(__FILE__, __LINE__, 2, "File '".$p_entry['filename']."' already exists");

      // ----- Look if file is a directory
      if (is_dir($p_entry['filename']))
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Existing file '".$p_entry['filename']."' is a directory");

        // ----- Change the file status
        $p_entry['status'] = "already_a_directory";

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
      // ----- Look if file is write protected
      else if (!is_writeable($p_entry['filename']))
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Existing file '".$p_entry['filename']."' is write protected");

        // ----- Change the file status
        $p_entry['status'] = "write_protected";

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Look if the extracted file is older
      else if (filemtime($p_entry['filename']) > $p_entry['mtime'])
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Existing file '".$p_entry['filename']."' is newer (".date("l dS of F Y h:i:s A", filemtime($p_entry['filename'])).") than the extracted file (".date("l dS of F Y h:i:s A", $p_entry['mtime']).")");

        // ----- Change the file status
        $p_entry['status'] = "newer_exist";

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
    }

    // ----- Check the directory availability and create it if necessary
    else
    {
      if (substr($p_entry['filename'], -1) == '/')
        $v_dir_to_check = $p_entry['filename'];
      else if (!strstr($p_entry['filename'], "/"))
        $v_dir_to_check = "";
      else
        $v_dir_to_check = dirname($p_entry['filename']);

      if (($v_result = $this->privDirCheck($v_dir_to_check, ($p_entry['external']==0x41FF0010))) != 1)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Unable to create path for '".$p_entry['filename']."'");

        // ----- Change the file status
        $p_entry['status'] = "path_creation_fail";

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
    }

    // ----- Do the extraction (if not a folder)
    if (!($p_entry['external']==0x41FF0010))
    {

      // ----- Look for not compressed file
      if ($p_entry['compressed_size'] == $p_entry['size'])
      {
        // ----- Trace
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Extracting an un-compressed file");

        // ----- Opening destination file
        if (($v_dest_file = @fopen($p_entry['filename'], 'wb')) == 0)
        {
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Error while opening '".$p_entry['filename']."' in write binary mode");

          // ----- Change the file status
          $p_entry['status'] = "write_error";

          // ----- Return
          PclTraceFctEnd(__FILE__, __LINE__, $v_result);
          return $v_result;
        }

        PclTraceFctMessage(__FILE__, __LINE__, 2, "Reading '".$p_entry['size']."' bytes");

        // ----- Read the file by PCLZIP_READ_BLOCK_SIZE octets blocks
        $v_size = $p_entry['compressed_size'];
        while ($v_size != 0)
        {
          $v_read_size = ($v_size < PCLZIP_READ_BLOCK_SIZE ? $v_size : PCLZIP_READ_BLOCK_SIZE);
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Read $v_read_size bytes");
          $v_buffer = fread($this->zip_fd, $v_read_size);
          $v_binary_data = pack('a'.$v_read_size, $v_buffer);
          @fwrite($v_dest_file, $v_binary_data, $v_read_size);
          $v_size -= $v_read_size;
        }

        // ----- Closing the destination file
        fclose($v_dest_file);

        // ----- Change the file mtime
        touch($p_entry['filename'], $p_entry['mtime']);
      }
      else
      {
        // ----- Trace
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Extracting a compressed file");

        // ----- Open the destination file in write mode
        if (($v_dest_file = @fopen($p_entry['filename'].'.gz', 'wb')) == 0)
        {
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Error while opening '".$p_entry['filename']."' in write binary mode");

          // ----- Change the file status
          $p_entry['status'] = "write_error";

          // ----- Return
          PclTraceFctEnd(__FILE__, __LINE__, $v_result);
          return $v_result;
        }

        PclTraceFctMessage(__FILE__, __LINE__, 2, "Start extraction of '".$p_entry['filename']."'");

        // ----- Write gz file format header
        $v_binary_data = pack('va1a1Va1a1', 0x8b1f, Chr($p_entry['compression']), Chr(0x00), time(), Chr(0x00), Chr(3));
        fwrite($v_dest_file, $v_binary_data, 10);

        // ----- Read the file by PCLZIP_READ_BLOCK_SIZE octets blocks
        $v_size = $p_entry['compressed_size'];
        while ($v_size != 0)
        {
          $v_read_size = ($v_size < PCLZIP_READ_BLOCK_SIZE ? $v_size : PCLZIP_READ_BLOCK_SIZE);
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Read $v_read_size bytes");
          $v_buffer = fread($this->zip_fd, $v_read_size);
          $v_binary_data = pack('a'.$v_read_size, $v_buffer);
          @fwrite($v_dest_file, $v_binary_data, $v_read_size);
          $v_size -= $v_read_size;
        }

        // ----- Write gz file format footer
        $v_binary_data = pack('VV', $p_entry['crc'], $p_entry['size']);
        fwrite($v_dest_file, $v_binary_data, 8);

        // ----- Close the temporary file
        fclose($v_dest_file);

        PclTraceFctMessage(__FILE__, __LINE__, 4, "Position after extract [".ftell($this->zip_fd)."]");

        // ----- Uncompress
        if (($v_src_file = gzopen($p_entry['filename'].'.gz', 'rb')) == 0)
        {
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Error while opening '".$p_entry['filename']."' in read binary mode");

          // ----- Change the file status
          $p_entry['status'] = "read_error";

          // ----- Return
          PclTraceFctEnd(__FILE__, __LINE__, $v_result);
          return $v_result;
        }

        if (($v_dest_file = @fopen($p_entry['filename'], 'wb')) == 0)
        {
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Error while opening '".$p_entry['filename']."' in write binary mode");

          // ----- Change the file status
          $p_entry['status'] = "write_error";
          gzclose($v_src_file);

          // ----- Return
          PclTraceFctEnd(__FILE__, __LINE__, $v_result);
          return $v_result;
        }

        PclTraceFctMessage(__FILE__, __LINE__, 2, 'File size is '.filesize($p_entry['filename'].'.gz'));
        PclTraceFctMessage(__FILE__, __LINE__, 2, "Reading '$p_entry[size]' bytes");

        // ----- Read the file by PCLZIP_READ_BLOCK_SIZE octets blocks
        $v_size = $p_entry['size'];
        while ($v_size != 0)
        {
          $v_read_size = ($v_size < PCLZIP_READ_BLOCK_SIZE ? $v_size : PCLZIP_READ_BLOCK_SIZE);
          PclTraceFctMessage(__FILE__, __LINE__, 2, "Read $v_read_size bytes");
          $v_buffer = gzread($v_src_file, $v_read_size);
          $v_binary_data = pack('a'.$v_read_size, $v_buffer);
          @fwrite($v_dest_file, $v_binary_data, $v_read_size);
          $v_size -= $v_read_size;
        }
        fclose($v_dest_file);
        gzclose($v_src_file);

        // ----- Change the file mtime
        touch($p_entry['filename'], $p_entry['mtime']);

        // ----- Delete the temporary file
        @unlink($p_entry['filename'].'.gz');
      }

      // ----- Trace
      PclTraceFctMessage(__FILE__, __LINE__, 2, "Extraction done");
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privReadFileHeader()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privReadFileHeader(&$p_header)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privReadFileHeader", "");
    $v_result=1;

    // ----- Read the 4 bytes signature
    $v_binary_data = @fread($this->zip_fd, 4);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Binary data is : '".sprintf("%08x", $v_binary_data)."'");
    $v_data = unpack('Vid', $v_binary_data);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Binary signature is : '".sprintf("0x%08x", $v_data['id'])."'");

    // ----- Check signature
    if ($v_data['id'] != 0x04034b50)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Invalid File header");

      // ----- Error log
      PclErrorLog(-10, 'Invalid archive structure');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Read the first 42 bytes of the header
    $v_binary_data = fread($this->zip_fd, 26);

    // ----- Look for invalid block size
    if (strlen($v_binary_data) != 26)
    {
      $p_header['filename'] = "";
      $p_header['status'] = "invalid_header";
      PclTraceFctMessage(__FILE__, __LINE__, 2, "Invalid block size : ".strlen($v_binary_data));

      // ----- Error log
      PclErrorLog(-10, "Invalid block size : ".strlen($v_binary_data));

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Extract the values
    PclTraceFctMessage(__FILE__, __LINE__, 2, "Header : '".$v_binary_data."'");
    PclTraceFctMessage(__FILE__, __LINE__, 2, "Header (Hex) : '".bin2hex($v_binary_data)."'");
    $v_data = unpack('vversion/vflag/vcompression/vmtime/vmdate/Vcrc/Vcompressed_size/Vsize/vfilename_len/vextra_len', $v_binary_data);

    // ----- Get filename
    PclTraceFctMessage(__FILE__, __LINE__, 3, "File name length : ".$v_data['filename_len']);
    $p_header['filename'] = fread($this->zip_fd, $v_data['filename_len']);
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Filename : \''.$p_header['filename'].'\'');

    // ----- Get extra_fields
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Extra field length : ".$v_data['extra_len']);
    if ($v_data['extra_len'] != 0) {
      $p_header['extra'] = fread($this->zip_fd, $v_data['extra_len']);
    }
    else {
      $p_header['extra'] = '';
    }
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Extra field : \''.bin2hex($p_header['extra']).'\'');

    // ----- Extract properties
    $p_header['compression'] = $v_data['compression'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Compression method : \''.bin2hex($p_header['compression']).'\'');
    $p_header['size'] = $v_data['size'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Size : \''.$p_header['size'].'\'');
    $p_header['compressed_size'] = $v_data['compressed_size'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Compressed Size : \''.$p_header['compressed_size'].'\'');
    $p_header['crc'] = $v_data['crc'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'CRC : \''.$p_header['crc'].'\'');
    $p_header['flag'] = $v_data['flag'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Flag : \''.$p_header['flag'].'\'');

    // ----- Recuperate date in UNIX format
    $p_header['mdate'] = $v_data['mdate'];
    $p_header['mtime'] = $v_data['mtime'];
    if ($p_header['mdate'] && $p_header['mtime'])
    {
      // ----- Extract time
      $v_hour = ($p_header['mtime'] & 0xF800) >> 11;
      $v_minute = ($p_header['mtime'] & 0x07E0) >> 5;
      $v_seconde = ($p_header['mtime'] & 0x001F)*2;

      // ----- Extract date
      $v_year = (($p_header['mdate'] & 0xFE00) >> 9) + 1980;
      $v_month = ($p_header['mdate'] & 0x01E0) >> 5;
      $v_day = $p_header['mdate'] & 0x001F;

      // ----- Get UNIX date format
      $p_header['mtime'] = mktime($v_hour, $v_minute, $v_seconde, $v_month, $v_day, $v_year);

      PclTraceFctMessage(__FILE__, __LINE__, 3, 'Date : \''.date("d/m/y H:i:s", $p_header['mtime']).'\'');
    }
    else
    {
      $p_header['mtime'] = time();
      PclTraceFctMessage(__FILE__, __LINE__, 3, 'Date is actual : \''.date("d/m/y H:i:s", $p_header['mtime']).'\'');
    }

    // ----- Other informations
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Compression type : ".$v_data['compression']);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Version : ".$v_data['version']);

    // TBC
    for(reset($v_data); $key = key($v_data); next($v_data)) {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Attribut[$key] = ".$v_data[$key]);
    }

    // ----- Set the stored filename
    $p_header['stored_filename'] = $p_header['filename'];

    // ----- Set the status field
    $p_header['status'] = "ok";

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privReadCentralFileHeader()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privReadCentralFileHeader(&$p_header)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privReadCentralFileHeader", "");
    $v_result=1;

    // ----- Read the 4 bytes signature
    $v_binary_data = @fread($this->zip_fd, 4);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Binary data is : '".sprintf("%08x", $v_binary_data)."'");
    $v_data = unpack('Vid', $v_binary_data);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Binary signature is : '".sprintf("0x%08x", $v_data['id'])."'");

    // ----- Check signature
    if ($v_data['id'] != 0x02014b50)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Invalid Central Dir File signature");

      // ----- Error log
      PclErrorLog(-10, 'Invalid archive structure');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Read the first 42 bytes of the header
    $v_binary_data = fread($this->zip_fd, 42);

    // ----- Look for invalid block size
    if (strlen($v_binary_data) != 42)
    {
      $p_header['filename'] = "";
      $p_header['status'] = "invalid_header";
      PclTraceFctMessage(__FILE__, __LINE__, 2, "Invalid block size : ".strlen($v_binary_data));

      // ----- Error log
      PclErrorLog(-10, "Invalid block size : ".strlen($v_binary_data));

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Extract the values
    PclTraceFctMessage(__FILE__, __LINE__, 2, "Header : '".$v_binary_data."'");
    PclTraceFctMessage(__FILE__, __LINE__, 2, "Header (Hex) : '".bin2hex($v_binary_data)."'");
    $p_header = unpack('vversion/vversion_extracted/vflag/vcompression/vmtime/vmdate/Vcrc/Vcompressed_size/Vsize/vfilename_len/vextra_len/vcomment_len/vdisk/vinternal/Vexternal/Voffset', $v_binary_data);

    // ----- Get filename
    PclTraceFctMessage(__FILE__, __LINE__, 3, "File name length : ".$p_header['filename_len']);
    if ($p_header['filename_len'] != 0)
      $p_header['filename'] = fread($this->zip_fd, $p_header['filename_len']);
    else
      $p_header['filename'] = '';
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Filename : \''.$p_header['filename'].'\'');

    // ----- Get extra
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Extra length : ".$p_header['extra_len']);
    if ($p_header['extra_len'] != 0)
      $p_header['extra'] = fread($this->zip_fd, $p_header['extra_len']);
    else
      $p_header['extra'] = '';
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Extra : \''.$p_header['extra'].'\'');

    // ----- Get comment
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Comment length : ".$p_header['comment_len']);
    if ($p_header['comment_len'] != 0)
      $p_header['comment'] = fread($this->zip_fd, $p_header['comment_len']);
    else
      $p_header['comment'] = '';
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Comment : \''.$p_header['comment'].'\'');

    // ----- Extract properties
    //$p_header['size'] = $v_data['size'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Size : \''.$p_header['size'].'\'');
    //$p_header['compressed_size'] = $v_data['compressed_size'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Compressed Size : \''.$p_header['compressed_size'].'\'');
    //$p_header['crc'] = $v_data['crc'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'CRC : \''.$p_header['crc'].'\'');
    //$p_header['flag'] = $v_data['flag'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Flag : \''.$p_header['flag'].'\'');
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Offset : \''.$p_header['offset'].'\'');

    // ----- Recuperate date in UNIX format
    if ($p_header['mdate'] && $p_header['mtime'])
    {
      // ----- Extract time
      $v_hour = ($p_header['mtime'] & 0xF800) >> 11;
      $v_minute = ($p_header['mtime'] & 0x07E0) >> 5;
      $v_seconde = ($p_header['mtime'] & 0x001F)*2;

      // ----- Extract date
      $v_year = (($p_header['mdate'] & 0xFE00) >> 9) + 1980;
      $v_month = ($p_header['mdate'] & 0x01E0) >> 5;
      $v_day = $p_header['mdate'] & 0x001F;

      // ----- Get UNIX date format
      $p_header['mtime'] = mktime($v_hour, $v_minute, $v_seconde, $v_month, $v_day, $v_year);

      PclTraceFctMessage(__FILE__, __LINE__, 3, 'Date : \''.date("d/m/y H:i:s", $p_header['mtime']).'\'');
    }
    else
    {
      $p_header['mtime'] = time();
      PclTraceFctMessage(__FILE__, __LINE__, 3, 'Date is actual : \''.date("d/m/y H:i:s", $p_header['mtime']).'\'');
    }

    // ----- Set the stored filename
    $p_header['stored_filename'] = $p_header['filename'];

    // ----- Set default status to ok
    $p_header['status'] = 'ok';

    // ----- Look if it is a directory
    PclTraceFctMessage(__FILE__, __LINE__, 2, "external (Hex) : '".sprintf("Ox%04X", $p_header['external'])."' (".($p_header['external']==0x41FF0010?'is a folder':'is a file').')');
    if (substr($p_header['filename'], -1) == '/')
    {
      $p_header['external'] = 0x41FF0010;
      PclTraceFctMessage(__FILE__, __LINE__, 3, 'Force folder external : \''.$p_header['external'].'\'');
    }


    // TBC
    for(reset($p_header); $key = key($p_header); next($p_header)) {
     PclTraceFctMessage(__FILE__, __LINE__, 3, "header[$key] = ".$p_header[$key]);
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privReadEndCentralDir()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privReadEndCentralDir(&$p_central_dir)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privReadEndCentralDir", "");
    $v_result=1;

    // ----- Go to the end of the zip file
    $v_size = filesize($this->zipname);
    PclTraceFctMessage(__FILE__, __LINE__, 2, "Size of the file :$v_size");
    @fseek($this->zip_fd, $v_size);
    PclTraceFctMessage(__FILE__, __LINE__, 4, 'Position at end of zip file : \''.ftell($this->zip_fd).'\'');
    if (@ftell($this->zip_fd) != $v_size)
    {
      // ----- Error log
      PclErrorLog(-10, 'Unable to go to the end of the archive \''.$this->zipname.'\'');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Go back to the maximum possible size of the Central Dir End Record
    $v_maximum_size = 277; // 0xFF + 22;
    if ($v_maximum_size > $v_size)
      $v_maximum_size = $v_size;
    @fseek($this->zip_fd, $v_size-$v_maximum_size);
    if (@ftell($this->zip_fd) != ($v_size-$v_maximum_size))
    {
      // ----- Error log
      PclErrorLog(-10, 'Unable to seek back to the middle of the archive \''.$this->zipname.'\'');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }
    PclTraceFctMessage(__FILE__, __LINE__, 4, 'Position after max central position : \''.ftell($this->zip_fd).'\'');

    // ----- Read byte per byte in order to find the signature
    $v_pos = ftell($this->zip_fd);
    $v_bytes = 0x00000000;
    while ($v_pos < $v_size)
    {
      // ----- Read a byte
      $v_byte = @fread($this->zip_fd, 1);

      // -----  Add the byte
      $v_bytes = ($v_bytes << 8) | Ord($v_byte);

      // ----- Compare the bytes
      if ($v_bytes == 0x504b0506)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 4, 'Found End Central Dir signature : \''.ftell($this->zip_fd).'\'');
        $v_pos++;
        break;
      }

      $v_pos++;
    }

    // ----- Look if not found end of central dir
    if ($v_pos == $v_size)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 2, "Unable to find End of Central Dir Record siganture");

      // ----- Error log
      PclErrorLog(0, "Unable to find End of Central Dir Record signature");

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Read the first 18 bytes of the header
    $v_binary_data = fread($this->zip_fd, 18);

    // ----- Look for invalid block size
    if (strlen($v_binary_data) != 18)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 2, "Invalid End of Central Dir Record size : ".strlen($v_binary_data));

      // ----- Error log
      PclErrorLog(0, "Invalid End of Central Dir Record size : ".strlen($v_binary_data));

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Extract the values
    //PclTraceFctMessage(__FILE__, __LINE__, 4, "Central Dir Record : '".$v_binary_data."'");
    //PclTraceFctMessage(__FILE__, __LINE__, 4, "Central Dir Record (Hex) : '".bin2hex($v_binary_data)."'");
    $v_data = unpack('vdisk/vdisk_start/vdisk_entries/ventries/Vsize/Voffset/vcomment_size', $v_binary_data);

    // ----- Check the global size
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Comment length : ".$v_data['comment_size']);
    if (($v_pos + $v_data['comment_size'] + 18) != $v_size)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 2, "Fail to find the right siganture");

      // ----- Error log
      PclErrorLog(-10, "Fail to find the right signature");

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Get comment
    if ($v_data['comment_size'] != 0)
      $p_central_dir['comment'] = fread($this->zip_fd, $v_data['comment_size']);
    else
      $p_central_dir['comment'] = '';
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Comment : \''.$p_central_dir['comment'].'\'');

    $p_central_dir['entries'] = $v_data['entries'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Nb of entries : \''.$p_central_dir['entries'].'\'');
    $p_central_dir['disk_entries'] = $v_data['disk_entries'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Nb of entries for this disk : \''.$p_central_dir['disk_entries'].'\'');
    $p_central_dir['offset'] = $v_data['offset'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Offset of Central Dir : \''.$p_central_dir['offset'].'\'');
    $p_central_dir['size'] = $v_data['size'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Size of Central Dir : \''.$p_central_dir['size'].'\'');
    $p_central_dir['disk'] = $v_data['disk'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Disk number : \''.$p_central_dir['disk'].'\'');
    $p_central_dir['disk_start'] = $v_data['disk_start'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, 'Start disk number : \''.$p_central_dir['disk_start'].'\'');

    // TBC
    //for(reset($p_central_dir); $key = key($p_central_dir); next($p_central_dir)) {
    //  PclTraceFctMessage(__FILE__, __LINE__, 3, "central_dir[$key] = ".$p_central_dir[$key]);
    //}

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : privDeleteByIndex()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function privDeleteByIndex($p_index, &$p_result_list)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privDeleteByIndex", "index='$p_index'");
    $v_result=1;
    $v_list_detail = array();

    // ----- Open the zip file
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary read mode");
    if (($v_result=$this->privOpenFd('rb')) != 1)
    {
      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Read the central directory informations
    $v_central_dir = array();
    if (($v_result = $this->privReadEndCentralDir($v_central_dir)) != 1)
    {
      $this->privCloseFd();
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Go to beginning of File
    PclTraceFctMessage(__FILE__, __LINE__, 5, "Position in file : ".ftell($this->zip_fd)."'");
    @rewind($this->zip_fd);
    PclTraceFctMessage(__FILE__, __LINE__, 5, "Position in file : ".ftell($this->zip_fd)."'");

    // ----- Scan all the files
    // ----- Manipulate the index list
    // TBC : In PHP4, we should have a char replace better than a string replace
    $p_index = str_replace(' ', '', $p_index);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Reduced index : '$p_index'");
    $v_index_list = explode(",", $p_index);
    sort($v_index_list);

    // ----- Start at beginning of Central Dir
    $v_pos_entry = $v_central_dir['offset'];
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Position before rewind : ".ftell($this->zip_fd)."'");
    @rewind($this->zip_fd);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after rewind : ".ftell($this->zip_fd)."'");
    if (@fseek($this->zip_fd, $v_pos_entry))
    {
      // ----- Close the zip file
      $this->privCloseFd();

      // ----- Error log
      PclErrorLog(-14, 'Invalid archive size');

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Position after fseek : ".ftell($this->zip_fd)."'");

    // ----- Read each entry
    $v_header_list = array();
    for ($i=0, $j_start=0, $v_nb_extracted=0; $i<$v_central_dir['entries']; $i++)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Read next file header entry : $i'");

      // ----- Read the file header
      $v_header_list[$v_nb_extracted] = array();
      if (($v_result = $this->privReadCentralFileHeader($v_header_list[$v_nb_extracted])) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();

        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Store the index
      $v_header_list[$v_nb_extracted]['index'] = $i;

      // ----- Look if the index is in the list
      for ($j=$j_start, $v_found = false; ($j<sizeof($v_index_list)) && (!$v_found); $j++)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 3, "Look if index '$i' is in '".$v_index_list[$j]."'");

        // ----- Extract range
        $v_item_list = explode("-", $v_index_list[$j]);
        $v_size_item_list = sizeof($v_item_list);
        if ($v_size_item_list == 1)
        {
          if ($i==$v_item_list[0])
          {
            PclTraceFctMessage(__FILE__, __LINE__, 3, "Found as a single index");
            $v_found = true;
          }
          if ($i>=$v_item_list[0])
          {
            $j_start = $j+1;
          }
        }
        else if ($v_size_item_list == 2)
        {
          if (($i>=$v_item_list[0]) && ($i<=$v_item_list[1]))
          {
            PclTraceFctMessage(__FILE__, __LINE__, 3, "Found as part of an index range");
            $v_found = true;
          }
          if ($i>=$v_item_list[1])
          {
            $j_start = $j+1;
          }
        }
        if ($v_item_list[0]>$i)
        {
          PclTraceFctMessage(__FILE__, __LINE__, 3, "Range is greater than index, stop loop");
          break;
        }
      }

      // ----- Look for deletion
      if ($v_found)
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "File '".$v_header['filename']."', index '$i' need to be deleted");
        unset($v_header_list[$v_nb_extracted]);
      }
      else
      {
        PclTraceFctMessage(__FILE__, __LINE__, 2, "File '".$v_header['filename']."', index '$i' will not be deleted");
        $v_nb_extracted++;
      }
    }

    // ----- Creates a temporay file
    $v_zip_temp_name = PCLZIP_TEMPORARY_DIR.uniqid('pclzip-').'.tmp';

    // ----- Creates a temporary zip archive
    $v_temp_zip = new PclZip($v_zip_temp_name);

    // ----- Open the temporary zip file in write mode
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Open file in binary write mode");
    if (($v_result = $v_temp_zip->privOpenFd('wb')) != 1)
    {
      $this->privCloseFd();

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Look which file need to be kept
    for ($i=0; $i<sizeof($v_header_list); $i++)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Keep entry : $i'");

      // ----- Calculate the position of the header
      PclTraceFctMessage(__FILE__, __LINE__, 2, "Offset='". $v_header_list[$i]['offset']."'");
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Position before rewind : ".ftell($this->zip_fd)."'");
      @rewind($this->zip_fd);
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Position after rewind : ".ftell($this->zip_fd)."'");
      if (@fseek($this->zip_fd,  $v_header_list[$i]['offset']))
      {
        // ----- Close the zip file
        $this->privCloseFd();
        $v_temp_zip->privCloseFd();
        @unlink($v_zip_temp_name);

        // ----- Error log
        PclErrorLog(-14, 'Invalid archive size');

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
        return PclErrorCode();
      }
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Position after fseek : ".ftell($this->zip_fd)."'");

      // ----- Read the file header
      if (($v_result = $this->privReadFileHeader($v_header_list[$i])) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();
        $v_temp_zip->privCloseFd();
        @unlink($v_zip_temp_name);

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Write the file header
      if (($v_result = $v_temp_zip->privWriteFileHeader($v_header_list[$i])) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();
        $v_temp_zip->privCloseFd();
        @unlink($v_zip_temp_name);

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
      PclTraceFctMessage(__FILE__, __LINE__, 3, "Offset for this file is '".$v_header_list[$i]['offset']."'");

      // ----- Read/write the data block
      if (($v_result = PclZipUtilCopyBlock($this->zip_fd, $v_temp_zip->zip_fd, $v_header_list[$i]['compressed_size'])) != 1)
      {
        // ----- Close the zip file
        $this->privCloseFd();
        $v_temp_zip->privCloseFd();
        @unlink($v_zip_temp_name);

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }
    }

    // ----- Store the offset of the central dir
    $v_offset = @ftell($v_temp_zip->zip_fd);
    PclTraceFctMessage(__FILE__, __LINE__, 4, "New offset of central dir : $v_offset");

    // ----- Re-Create the Central Dir files header
    for ($i=0; $i<sizeof($v_header_list); $i++)
    {
      // ----- Create the file header
      PclTraceFctMessage(__FILE__, __LINE__, 4, "Offset of file : ".$v_header_list[$i]['offset']);
      if (($v_result = $v_temp_zip->privWriteCentralFileHeader($v_header_list[$i])) != 1)
      {
        $v_temp_zip->privCloseFd();
        $this->privCloseFd();
        @unlink($v_zip_temp_name);

        // ----- Return
        PclTraceFctEnd(__FILE__, __LINE__, $v_result);
        return $v_result;
      }

      // ----- Transform the header to a 'usable' info
      $v_temp_zip->privConvertHeader2FileInfo($v_header_list[$i], $p_result_list[$i]);
    }

    // ----- Zip file comment
    $v_comment = '';

    // ----- Calculate the size of the central header
    $v_size = @ftell($v_temp_zip->zip_fd)-$v_offset;

    // ----- Create the central dir footer
    if (($v_result = $v_temp_zip->privWriteCentralHeader(sizeof($v_header_list), $v_size, $v_offset, $v_comment)) != 1)
    {
      // ----- Reset the file list
      unset($v_header_list);
      $v_temp_zip->privCloseFd();
      $this->privCloseFd();
      @unlink($v_zip_temp_name);

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, $v_result);
      return $v_result;
    }

    // ----- Close
    $v_temp_zip->privCloseFd();
    $this->privCloseFd();

    // ----- Delete the zip file
    // TBC : I should test the result ...
    @unlink($this->zipname);

    // ----- Rename the temporary file
    // TBC : I should test the result ...
    //@rename($v_zip_temp_name, $this->zipname);
    PclZipUtilRename($v_zip_temp_name, $this->zipname);

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------


  // --------------------------------------------------------------------------------
  // Function : privDirCheck()
  // Description :
  //   Check if a directory exists, if not it creates it and all the parents directory
  //   which may be useful.
  // Parameters :
  //   $p_dir : Directory path to check.
  // Return Values :
  //    1 : OK
  //   -1 : Unable to create directory
  // --------------------------------------------------------------------------------
  function privDirCheck($p_dir, $p_is_dir=false)
  {
    $v_result = 1;

    PclTraceFctStart(__FILE__, __LINE__, "PclZip::privDirCheck", "entry='$p_dir', is_dir='".($p_is_dir?"true":"false")."'");

    // ----- Remove the final '/'
    if (($p_is_dir) && (substr($p_dir, -1)=='/'))
    {
      $p_dir = substr($p_dir, 0, strlen($p_dir)-1);
    }
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Looking for entry '$p_dir'");

    // ----- Check the directory availability
    if ((is_dir($p_dir)) || ($p_dir == ""))
    {
      PclTraceFctEnd(__FILE__, __LINE__, "'$p_dir' is a directory");
      return 1;
    }

    // ----- Extract parent directory
    $p_parent_dir = dirname($p_dir);
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Parent directory is '$p_parent_dir'");

    // ----- Just a check
    if ($p_parent_dir != $p_dir)
    {
      // ----- Look for parent directory
      if ($p_parent_dir != "")
      {
        if (($v_result = $this->privDirCheck($p_parent_dir)) != 1)
        {
          PclTraceFctEnd(__FILE__, __LINE__, $v_result);
          return $v_result;
        }
      }
    }

    // ----- Create the directory
    PclTraceFctMessage(__FILE__, __LINE__, 3, "Create directory '$p_dir'");
    if (!@mkdir($p_dir, 0777))
    {
      // ----- Error log
      PclErrorLog(-8, "Unable to create directory '$p_dir'");

      // ----- Return
      PclTraceFctEnd(__FILE__, __LINE__, PclErrorCode(), PclErrorString());
      return PclErrorCode();
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result, "Directory '$p_dir' created");
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  }
  // End of class
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : PclZipUtilPathReduction()
  // Description :
  // Parameters :
  // Return Values :
  // --------------------------------------------------------------------------------
  function PclZipUtilPathReduction($p_dir)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZipUtilPathReduction", "dir='$p_dir'");
    $v_result = "";

    // ----- Look for not empty path
    if ($p_dir != "")
    {
      // ----- Explode path by directory names
      $v_list = explode("/", $p_dir);

      // ----- Study directories from last to first
      for ($i=sizeof($v_list)-1; $i>=0; $i--)
      {
        // ----- Look for current path
        if ($v_list[$i] == ".")
        {
          // ----- Ignore this directory
          // Should be the first $i=0, but no check is done
        }
        else if ($v_list[$i] == "..")
        {
          // ----- Ignore it and ignore the $i-1
          $i--;
        }
        else if (($v_list[$i] == "") && ($i!=(sizeof($v_list)-1)) && ($i!=0))
        {
          // ----- Ignore only the double '//' in path,
          // but not the first and last '/'
        }
        else
        {
          $v_result = $v_list[$i].($i!=(sizeof($v_list)-1)?"/".$v_result:"");
        }
      }
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : PclZipUtilCopyBlock()
  // Description :
  // Parameters :
  //   $p_mode : read/write compression mode
  //             0 : src & dest normal
  //             1 : src gzip, dest normal
  //             2 : src normal, dest gzip
  //             3 : src & dest gzip
  // Return Values :
  // --------------------------------------------------------------------------------
  function PclZipUtilCopyBlock($p_src, $p_dest, $p_size, $p_mode=0)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZipUtilCopyBlock", "size=$p_size, mode=$p_mode");
    $v_result = 1;

    if ($p_mode==0)
    {
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Src offset before read :".(@ftell($p_src)));
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Dest offset before write :".(@ftell($p_dest)));
      while ($p_size != 0)
      {
        $v_read_size = ($p_size < PCLZIP_READ_BLOCK_SIZE ? $p_size : PCLZIP_READ_BLOCK_SIZE);
        PclTraceFctMessage(__FILE__, __LINE__, 4, "Read $v_read_size bytes");
        $v_buffer = @fread($p_src, $v_read_size);
        @fwrite($p_dest, $v_buffer, $v_read_size);
        $p_size -= $v_read_size;
      }
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Src offset after read :".(@ftell($p_src)));
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Dest offset after write :".(@ftell($p_dest)));
    }
    else if ($p_mode==1)
    {
      while ($p_size != 0)
      {
        $v_read_size = ($p_size < PCLZIP_READ_BLOCK_SIZE ? $p_size : PCLZIP_READ_BLOCK_SIZE);
        PclTraceFctMessage(__FILE__, __LINE__, 4, "Read $v_read_size bytes");
        $v_buffer = @gzread($p_src, $v_read_size);
        @fwrite($p_dest, $v_buffer, $v_read_size);
        $p_size -= $v_read_size;
      }
    }
    else if ($p_mode==2)
    {
      while ($p_size != 0)
      {
        $v_read_size = ($p_size < PCLZIP_READ_BLOCK_SIZE ? $p_size : PCLZIP_READ_BLOCK_SIZE);
        PclTraceFctMessage(__FILE__, __LINE__, 4, "Read $v_read_size bytes");
        $v_buffer = @fread($p_src, $v_read_size);
        @gzwrite($p_dest, $v_buffer, $v_read_size);
        $p_size -= $v_read_size;
      }
    }
    else if ($p_mode==3)
    {
      while ($p_size != 0)
      {
        $v_read_size = ($p_size < PCLZIP_READ_BLOCK_SIZE ? $p_size : PCLZIP_READ_BLOCK_SIZE);
        PclTraceFctMessage(__FILE__, __LINE__, 4, "Read $v_read_size bytes");
        $v_buffer = @gzread($p_src, $v_read_size);
        @gzwrite($p_dest, $v_buffer, $v_read_size);
        $p_size -= $v_read_size;
      }
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------

  // --------------------------------------------------------------------------------
  // Function : PclZipUtilRename()
  // Description :
  //   This function tries to do a simple rename() function. If it fails, it
  //   tries to copy the $p_src file in a new $p_dest file and then unlink the
  //   first one.
  // Parameters :
  //   $p_src : Old filename
  //   $p_dest : New filename
  // Return Values :
  //   1 on success, 0 on failure.
  // --------------------------------------------------------------------------------
  function PclZipUtilRename($p_src, $p_dest)
  {
    PclTraceFctStart(__FILE__, __LINE__, "PclZipUtilRename", "source=$p_src, destination=$p_dest");
    $v_result = 1;

    // ----- Try to rename the files
    if (!@rename($p_src, $p_dest)) {
      PclTraceFctMessage(__FILE__, __LINE__, 5, "Fail to rename file, try copy+unlink");

      // ----- Try to copy & unlink the src
      if (!@copy($p_src, $p_dest)) {
        PclTraceFctMessage(__FILE__, __LINE__, 5, "Fail to copy file");
        $v_result = 0;
      }
      else if (!@unlink($p_src)) {
        PclTraceFctMessage(__FILE__, __LINE__, 5, "Fail to unlink old filename");
        $v_result = 0;
      }
    }

    // ----- Return
    PclTraceFctEnd(__FILE__, __LINE__, $v_result);
    return $v_result;
  }
  // --------------------------------------------------------------------------------


// ----- End of double include look
}
?>